/**
* \file: EventDispatcher.cpp
*
* \version: 0.1
*
* \release: $Name:$
*
* Implementation to test and control the FeatureDisovery.
*
* \component: BDCL
*
* \author: D. Girnus / ADIT/SW2 / dgirnus@de.adit-jv.com
*         P. Govindaraju/ pradeepa.govindaraju@in.bosch.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <sys/prctl.h>

/* DLT logging */
#include <adit_logging.h>
#include "EventDispatcher.h"

LOG_IMPORT_CONTEXT(tbdcl)

using namespace adit::uspi;

namespace adit { namespace bdcl {



EventDispatcher::EventDispatcher()
{
    mEventDispatcherThreadId = -1;
    mRunning    = false;
    mSemCreated = false;
}

EventDispatcher::~EventDispatcher()
{
    LOG_INFO((tbdcl, "EventDispatcher::%s()  called", __func__));
    stopEventDispatcher();
}

bool EventDispatcher::startEventDispatcher()
{
    bool res = false;

    mRunning = true;

    int32_t err = sem_init(&mEventAvailable, 0, 0);
    if (err == -1) {
        LOG_ERROR((tbdcl, "EventDispatcher::%s()  create Semaphore failed err=%d, errno=%d",
                __func__, err, errno));
        res = false;
    } else {
        mSemCreated = true;
    }

    /* create thread */
    err = pthread_create(&mEventDispatcherThreadId, nullptr, &EventDispatcherThread, this);
    if (0 != err) {
        LOG_ERROR((tbdcl, "EventDispatcher::%s()  create FdTestEventHandlerThread failed err=%d, errno=%d",
                __func__, err, errno));
        res = false;
    } else {
        LOG_INFO((tbdcl, "EventDispatcher::%s()  EventDispatcherThread created", __func__));
        res = true;
    }
    return res;
}


void EventDispatcher::stopEventDispatcher()
{
    bool wasRunning = mRunning;
    mRunning = false;

    if (false == wasRunning) {
        LOG_INFO((tbdcl, "EventDispatcher::%s()  already stopped", __func__));
        return;
    }

    if (true == mSemCreated) {
        /* this should wake-up the EventHandlerThread */
        sem_post(&mEventAvailable);
    }

    if (mEventDispatcherThreadId > 0) {
        int32_t err = pthread_join(mEventDispatcherThreadId, nullptr);
        LOG_INFO((tbdcl, "EventDispatcher::%s()  join EventDispatcherThread returned with err=%d, errno=%d",
                __func__, err, errno));
        mEventDispatcherThreadId = -1;
    }

    if (true == mSemCreated) {
        sem_destroy(&mEventAvailable);
        LOG_INFO((tbdcl, "EventDispatcher::%s()  sem_destroy done", __func__));
        mSemCreated = false;
    }
}

void* EventDispatcher::EventDispatcherThread(void* context)
{

    int res = 0;
    auto me = static_cast<EventDispatcher*>(context);
    if (me == nullptr) {
        LOG_ERROR((tbdcl, "EventDispatcher::%s()  could not cast input pointer", __func__));
        return nullptr; /* ====== LEAVE FUNCTION ====== */
    }

    prctl(PR_SET_NAME,"Eventdispatch",0,0,0);

    while (me->mRunning)
    {
        std::shared_ptr<EventItem> event = me->dequeueEvent();
        if ((event == nullptr) || (false == me->mRunning)) {

            LOG_WARN((tbdcl, "EventDispatcher::%s()  event is NULL or mRunning is false ", __func__));
            // Happens while shutting down.
            break;
        }

        /* handle event */
        res = me->handleEvent(event);
    }

    LOG_INFO((tbdcl, "EventDispatcher::%s()  stop EventDispatcher (res=%d)", __func__, res));

    return nullptr;
}

int EventDispatcher::handleEvent(std::shared_ptr<EventItem> inEventItem)
{
    uint32_t eventMask = 0;
    int res = 0;
    EventType eventType = inEventItem->getEventType();

    if (EventType::FOUND_APPLE_DEVICE == eventType) {
        std::shared_ptr<DiscoveredDeviceUsb> pUsbDevice = std::dynamic_pointer_cast<DiscoveredDeviceUsb>(inEventItem->getDevice());
        LOG_INFO((tbdcl, "EventDispatcher::%s() onAppleDeviceFound ", __func__));
        onAppleDeviceFound(pUsbDevice);
    } else if (EventType::FOUND_AOAP_DEVICE == eventType){
        std::shared_ptr<DiscoveredDeviceUsb> pUsbDevice = std::dynamic_pointer_cast<DiscoveredDeviceUsb>(inEventItem->getDevice());
        LOG_INFO((tbdcl, "EventDispatcher::%s() onAoapDeviceFound ", __func__));
        onAoapDeviceFound(pUsbDevice);

    } else if (EventType::LOST_APPLE_DEVICE == eventType) {
        std::shared_ptr<DiscoveredDeviceUsb> pUsbDevice = std::dynamic_pointer_cast<DiscoveredDeviceUsb>(inEventItem->getDevice());
        LOG_INFO((tbdcl, "EventDispatcher::%s() onAppleDeviceLost ", __func__));
        onAppleDeviceLost(pUsbDevice);

    } else if (EventType::LOST_AOAP_DEVICE == eventType) {
        std::shared_ptr<DiscoveredDeviceUsb> pUsbDevice = std::dynamic_pointer_cast<DiscoveredDeviceUsb>(inEventItem->getDevice());
        LOG_INFO((tbdcl, "EventDispatcher::%s() onAoapDeviceLost ", __func__));
        onAoapDeviceLost(pUsbDevice);

    } else if (EventType::SWITCHED == eventType) {

        std::shared_ptr<DiscoveredDeviceUsb> pUsbDevice = std::dynamic_pointer_cast<DiscoveredDeviceUsb>(inEventItem->getDevice());
        /* get event mask from discovered device object */
        eventMask = pUsbDevice->getEventMask();

        if (eventMask == DD_USB_AOAP) {
            onDeviceSwitched(pUsbDevice);
            LOG_INFO((tbdcl, "EventDispatcher::%s() onDeviceSwitched ", __func__));
        } else {
            LOG_WARN((tbdcl, "EventDispatcher::%s()  switchedCb() received but eventMask (%u) does not mention AOAP device ", __func__, eventMask));
        }
    } else if (EventType::CHANGED == eventType) {

        std::shared_ptr<DiscoveredDeviceUsb> pUsbDevice = std::dynamic_pointer_cast<DiscoveredDeviceUsb>(inEventItem->getDevice());
        LOG_INFO((tbdcl, "EventDispatcher::%s() onDeviceChanged ", __func__));
        onDeviceChanged(pUsbDevice);

    } else if (EventType::EANSTART == eventType) {
        LOG_INFO((tbdcl, "EventDispatcher::%s() onStartEANative ", __func__));
        onStartEANative(inEventItem->getEaWriteFile() , inEventItem->getEaReadFile() );

    } else if (EventType::EANSTOP == eventType) {
        LOG_INFO((tbdcl, "EventDispatcher::%s() onStopEANative ", __func__));
        onStopEANative(inEventItem->getEaWriteFile() , inEventItem->getEaReadFile() );

    } else if (EventType::CMDMDEXIT == eventType) {
        LOG_INFO((tbdcl, "EventDispatcher::%s() onMDExit ", __func__));
        onMDExit("On CMDMDEXIT");

    }else if (EventType::AOAPREADTIMEOUT == eventType) {
        LOG_INFO((tbdcl, "EventDispatcher::%s() onAOAPReadTimeOut ", __func__));
        onAoapReadTimeOutError(inEventItem->getEaReadFile());
    }else {
        LOG_ERROR((tbdcl, "EventDispatcher::%s()  unknown event %d",
                __func__, (int32_t)eventType));
        res = -1;
    }
    return res;
}

void EventDispatcher::queueEvent(const std::shared_ptr<EventItem>& work)
{
    if (mRunning) {
        {
            std::lock_guard<std::mutex> lock(mEventQueueMutex);

            mEventQueue.push_back(work);

            LOG_INFO((tbdcl, "EventDispatcher::%s()  No. of events in list:  %zd", __func__, mEventQueue.size()));
        } /*mEventQueueMutex is automatically  released here when lock goes out of scope*/

        sem_post(&mEventAvailable);
    } else {
        LOG_INFO((tbdcl, "EventDispatcher::%s()  mRunning was set, event discarded", __func__));
    }
}

std::shared_ptr<EventItem> EventDispatcher::dequeueEvent()
{
    if (mRunning) {
        sem_wait(&mEventAvailable);
        LOG_INFO((tbdcl, "EventDispatcher::%s()  sem_wait released", __func__));
    } else {
        LOG_INFO((tbdcl, "EventDispatcher::%s()  mRunning was set, ignore sem_wait", __func__));
    }

    std::shared_ptr<EventItem> event(nullptr);
    {
        std::lock_guard<std::mutex> lock(mEventQueueMutex);

        /* lock access to queue here! */
        if (mEventQueue.size() == 0) {
            LOG_INFO((tbdcl, "EventDispatcher::%s()  mEventQueue is empty!", __func__));
            return nullptr;
        }
        event = mEventQueue.front();
        mEventQueue.pop_front();
    }/*mEventQueueMutex is automatically  released here when lock goes out of scope*/

    return event;
}

} } // namespace adit { namespace uspi {
